/***********************************************************************************************\
* Freescale MMA845xQ Driver
*
* Filename: terminal.c
*
*
* (c) Copyright 2011, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
\***********************************************************************************************/

#include "system.h"

/***********************************************************************************************\
* Private macros
\***********************************************************************************************/

/***********************************************************************************************\
* Private type definitions
\***********************************************************************************************/


/***********************************************************************************************\
* Private prototypes
\***********************************************************************************************/

//void Check800Hz (void);
void CheckHPF (byte);
byte ProcessHexInput (byte *in, byte *out);
void CopyXYZ (byte *ptr);
void PrintXYZdec8 (void);
//void PrintXYZdec10 (void);
//void PrintXYZdec12 (void);
void PrintXYZdec14 (void);
void PrintXYZdec (void);
void PrintXYZfrac (void);
void ClearDataFlash (void);
void PrintBuffering (void);
void PrintFIFO (void);
//void ReadDataFlashXYZ (void);
//void WriteFlashBuffer1 (void);
//void WriteFlashBuffer2 (void);

/***********************************************************************************************\
* Private memory declarations
\***********************************************************************************************/

#pragma DATA_SEG __SHORT_SEG _DATA_ZEROPAGE

BIT_FIELD StreamMode;                       // stream mode control flags

extern BIT_FIELD RegisterFlag;
extern byte value[];                        // working value result scratchpad

static byte temp;                           // temporary byte variable
static byte actv;                           // temporary location to store active state
static byte index;                          // input character string index

extern byte SlaveAddressIIC;                // accelerometer slave I2C address
//static byte deviceID;                       // contents from who_am_i register

extern byte functional_block;               // accelerometer function
                                        
extern byte address_in[3];                // Data Flash input address pointer
extern byte address_out[3];               // Data Flash output address pointer

#pragma DATA_SEG DEFAULT

static tword x_acc_value;                   // X accelerometer value
static tword y_acc_value;                   // Y accelerometer value
static tword z_acc_value;                   // Z accelerometer value

static tword x_mag_value;                   // X magnetometer value
static tword y_mag_value;                   // Y magnetometer value
static tword z_mag_value;                   // Z magnetometer value

extern tfifo_sample fifo_data[FIFO_BUFFER_SIZE];   // FIFO sample buffer

// operating modes (TO, Aug 2012)

byte *const OperatingModevalues[] =
{
  "Accel Only", "Mag Only", "Rsrvd", "Hybrid (Accel+Mag)"
};

byte *const AccelOSRModevalues[] =
{
  "Normal", "Low Noise", "High Res", "Low Power"
};

// Look up table for mag OSR (TO, Aug 2012)

byte const MagOSRLookUp [8][8] =
{
  { 0, 0, 0, 0, 0, 0, 0, 0 }, // ODR=800 
  { 0, 0, 0, 0, 0, 0, 0, 1 }, // ODR=400
  { 0, 0, 0, 0, 0, 0, 1, 2 }, // ODR=200 
  { 0, 0, 0, 0, 0, 1, 2, 3 }, // ODR=100
  { 0, 0, 0, 0, 1, 2, 3, 4 }, // ODR=50
  { 0, 0, 1, 2, 3, 4, 5, 6 }, // ODR=12.5
  { 1, 1, 2, 3, 4, 5, 6, 7 }, // ODR=6.25
  { 3, 3, 4, 5, 6, 7, 8, 9 }  // ODR=1.56 or half for all    
};

// mag OSR values (TO, Aug 2012)

byte *const MagOSRvalues [] = 
{
  "2", "4", "8", "16", "32", "64", "128", "256", "512", "1024"
};

byte *const ODRvalues [] = 
{
  "800", "400", "200", "100", "50", "12.5", "6.25", "1.56"
};

// hyb ODR values (TO, Aug 2012)

byte *const HybridODRvalues [] = 
{
  "400", "200", "100", "50", "25", "6.25", "3.15", "0.8"
};

byte const HPFlookup [4][8] =
{
  { 0, 0, 1, 2, 3, 3, 3, 3 }, // Normal mode
  { 0, 0, 1, 2, 3, 5, 5, 5 }, // Low Noise mode
  { 0, 0, 0, 0, 0, 0, 0, 0 }, // High Res mode
  { 0, 1, 2, 3, 4, 6, 6, 6 }  // low Power mode
};

byte *const HPvalues [] = 
{
  "16", "8", "4", "2", "1", "0.5", "0.25", "0.125", "0.063", "0.031"
};

// (TO, Aug 2012), no need for Stby here...

byte *const GRange [] =
{
  /*"Standby",*/ "2g", "4g", "8g", "Rsvd"
};


/***********************************************************************************************\
* Public functions
\***********************************************************************************************/

/*********************************************************\
**  Terminal Strings
\*********************************************************/

const byte string_PromptX []      = {"\r\nFXOS8700CQ> "};
const byte string_What []         = {" <-- what?"};

const byte string_Streaming []    = {" - Streaming XYZ data"};
const byte string_Counts []       = {" as signed counts\r\n"};
const byte string_Gs []           = {" in signed units\r\n"};
const byte string_Interrupts []   = {" via INT"};
const byte string_FIFO []         = {" via FIFO"};

const byte string_a_N []          = {" a=N Normal data\r\n"};
const byte string_a_NH []         = {" a=N Normal data; a=H HPF data\r\n"};
const byte string_aa_N []         = {" CN=counts Normal, GN=gs Normal\r\n"};
const byte string_aa_NH []        = {" CN=counts Normal, CH=counts HPF, GN=g's Normal, GH=g's HPF\r\n"};

const byte string_warn_banner []     = {"*******************************************************************\r\n"};
//const byte string_warn_slow_flash [] = {"* WARNING - The flash memory may not be reliable with 800 Hz data *\r\n"};
const byte string_warn_hpf []        = {"*  WARNING - HPF requires samples to settle after an activation   *\r\n"};
const byte string_prompt_spacer []   = {"            "};

const byte string_warn_slow_streaming [] = {"\r\n\r\n*Recommended Streaming ODR<=100Hz for HYBRID mode,*\r\n*ODR<=200Hz for Mag Only or Accel Only modes*\r\n\r\n"};
const byte string_warn_ctrl_reg1 [] = {"\r\n*Part must be in STBY before changing ODR*\r\n"};

// Software Rev String, (TO, Aug 2012)

const byte string_software_rev [] = {"1.20"}; 

const byte *string_Prompt;
const byte *string_a;
const byte *string_aa;

/*********************************************************\
**  Initialize Terminal Interface
\*********************************************************/
void TerminalInit (void)
{
  SCISendString("\r\n\n**  Freescale Semiconductor  **");
  SCISendString  ("\r\n**  FXOS8700CQ Demo          **");
  SCISendString  ("\r\n**      using the MC9S08QE8  **");
  SCISendString  ("\r\n**                           **");
  SCISendString  ("\r\n**  "__DATE__"    "__TIME__"  **\r\n\n");
  
  /*
  **  Read Flash.ID : SST25VF032B
  */
  
  // (TO, Aug 2012)
  //SPI_SS_SELECT;
  //SPI_ChrShift(0x9F);
  //SPI_ChrShift(0x00);
  //SPI_ChrShift(0x00);
  //SPI_ChrShift(0x00);
  //value[0] = SPI_ChrShiftR(0x00);
  //value[1] = SPI_ChrShiftR(0x00);
  //value[2] = SPI_ChrShiftR(0x00);
  //SPI_SS_DESELECT;
  
  //SCISendString("Data Flash JEDEC ID : Manf=");
  //SCI_ByteOut(value[0]);
  //SCISendString(" Type=");
  //SCI_ByteOut(value[1]);
  //SCISendString(" Capacity=");
  //SCI_ByteOut(value[2]);
  //if (value[0]==0xBF && value[1]==0x25 && value[2]==0x4A)
    //SCISendString(" : SST25VF032B");
  //else
    //SCISendString(" : unknown");
  
  SCISendString("Command Line Interface Software Revision: ");
  SCISendString(string_software_rev);
  SCI_putCRLF();
  SCI_putCRLF();

  SCISendString("This version of command line interface does");
  SCI_putCRLF();
  SCISendString("not support external flash");
  SCI_putCRLF();
  
  /*
  **  Prepare Data Flash.
  */
//  SCISendString  ("Erasing Data Flash\r");
//  while (SCItxActive);
  
  // (TO, Aug 2012)
  //DATAFLASH_Unprotect();
  
  address_in[0] = 0;
  address_in[1] = 0;
  address_in[2] = 0;
  address_out[0] = 0;
  address_out[1] = 0;
  address_out[2] = 0;
  // Flash - Write Status Register - Clears Write Enable Latch
  //SPI_SS_SELECT;
  //SPI_ChrShift(0x01);
  //SPI_ChrShift(0x00);
  //SPI_SS_DESELECT;
//  ClearDataFlash();
  
  functional_block = FBID_MAX;
  StreamMode.Byte = 0;
}

/*********************************************************\
**  Set Terminal Prompt
\*********************************************************/
void TerminalSetPrompt (byte devID)
{
  string_Prompt = string_PromptX; 
  
}

/*********************************************************\
**  Process Terminal Interface
\*********************************************************/

void ProcessTerminal (void)
{
  /*
  **  Output command prompt if required.
  */
  if (PROMPT_WAIT == FALSE)
  {
    SCISendString(string_Prompt);
    PROMPT_WAIT = TRUE;
  }
  /*
  **  Get input from terminal.
  */
  if (SCI_INPUT_READY == TRUE)
  {
    INPUT_ERROR = FALSE;
    /*
    **  Use command line input if not streaming data
    */
    if ((XYZ_STREAM == FALSE) && (INT_STREAM == FALSE))
    {
      PROMPT_WAIT = FALSE;
      /*
      **  Get first input character - only uppercase
      */
      switch (BufferRx[0])
      {
        /***************************************************************************************/
        case '?':
          /*
          **  Help : list valid commands
          */
          SCISendString("List of FXOS8700CQ commands:\r\n");
          //SCISendString("Mn       : Mode 1=Standby; 2=2g; 4=4g; 8=8g\r\n");
          //SCISendString("On       : Oversampling 0=Normal; 1=LNLP; 2=HighRes; 3=LP\r\n");
          //SCISendString("RO n     : ODR Hz 0=800; 1=400; 2=200; 3=100; 4=50; 5=12.5; 6=6.25; 7=1.56\r\n");
          SCISendString("RR xx    : Register xx Read\r\n");
          SCISendString("RW xx=nn : Register xx Write value nn\r\n");
          SCISendString("RF       : Report Configuration Specifics\r\n");
          SCISendString("CN       : XYZ data as Signed Counts\r\n");
          SCISendString("CH       : XYZ data as Signed Counts, with Accel HP Filter enabled\r\n");
          //SCISendString(string_a);
          SCISendString("GN       : XYZ data in Signed Units\r\n");
          SCISendString("GH       : XYZ data in Signed Units, with Accel HP Filter enabled\r\n");
          //SCISendString(string_a);
          //SCISendString("P a      : XYZ data as signed SI units with orientation (P/L)");
          //SCISendString(string_a);
          SCISendString("S aa     : Stream XYZ:\r\n");
          //SCISendString("         :  aa:");
          //SCISendString(string_aa);
          SCISendString("I aa n   : Stream XYZ via Interrups\r\n");
          //SCISendString("         :  aa:");
          //SCISendString(string_aa);
          SCISendString("         :  n:  1=INT1; 2=INT2\r\n");
          SCISendString("F aa ww  : Stream XYZ via FIFO\r\n");
          //SCISendString("         :  aa:");
          //SCISendString(string_aa);
          SCISendString("         :  aa: CN, CH, GN or GH as explained above\r\n");
          SCISendString("         :  ww: Watermark= 1 to 31\r\n");
          break;

        /***************************************************************************************/
        // TO DO: Do not need it, change with ACTIVE/STBY command
        // (TO, Aug 2012)
//        case 'M':
          /*

          **  Control sensor mode of operation
          */
//          switch (BufferRx[1])
//          {
            /////////////////////////////////////////////////////////////////////////////////////
//            case '0':
              /*
              **  M0  : Enter Shutdown
              */
//              SCISendString(" - Sensor Shutdown");
//              SENSOR_SHUTDOWN;
//              break;

            /////////////////////////////////////////////////////////////////////////////////////
//            case '1':
              /*
              **  M1  : Enter Standby
              */
//              SCISendString(" - Sensor Standby");
//              SENSOR_ACTIVE;
//              (void)FXOS8700CQ_Standby();
//              break;

            /////////////////////////////////////////////////////////////////////////////////////
//            case '2':
               /***************************************************************************
                ** M2  : Enter Active 2g
                ** Code Location: terminal.c
               ***************************************************************************/

//              SCISendString(" - Sensor Active 2g");
                //Get bit values for 2g mode 
//              SENSOR_ACTIVE;
              /*
              ** Enter Standby, CTRL_REG1 ACTIVE bit = 0 
              */
//              (void)FXOS8700CQ_Standby();
              /*
              ** Read the contents of the XYZ_DATA_CFG_REG
              */     
//              temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);    
              /*
              ** Mask the bits to calculate the value to write to the XYZ_DATA_CFG_REG to enable 2g mode
              */
//              full_scale = FULL_SCALE_2G;    // FULL_SCALE_2G = 0x00
//              temp = (temp & (~FS_MASK)) | full_scale;  
              /*
              ** Write the value back into XYZ_DATA_CFG_REG
              */
//              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              /*
              ** Set device into Active Mode
              **
              */
//              FXOS8700CQ_Active();
//              break;

            /////////////////////////////////////////////////////////////////////////////////////
//            case '4':
               /***************************************************************************
                ** M4  : Enter Active 4g
                ** Code Location: terminal.c
               ***************************************************************************/
//              SCISendString(" - Sensor Active 4g");
              
//              SENSOR_ACTIVE;
              /*
              ** Enter Standby, CTRL_REG1 ACTIVE bit = 0 
              */
//              (void)FXOS8700CQ_Standby();
              /*
              ** Read the contents of the XYZ_DATA_CFG_REG
              */
//              temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
              /*
              ** Mask the bits to calculate the value to write to the XYZ_DATA_CFG_REG to enable 4g mode
              */
//              full_scale = FULL_SCALE_4G;   // FULL_SCALE_4G = 0x01
//              temp = (temp & (~FS_MASK)) | full_scale;
              /*
              ** Write the value back into XYZ_DATA_CFG_REG
              */
//              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              /*
              ** Set device into Active Mode
              **
              */
//              FXOS8700CQ_Active();
//              break;

            /////////////////////////////////////////////////////////////////////////////////////
//            case '8':
             /***************************************************************************
             ** M8  : Enter Active 8g
             ** Code Location: terminal.c
             ***************************************************************************/
//              SCISendString(" - Sensor Active 8g");
              
//              SENSOR_ACTIVE;
              /*
              ** Enter Standby, CTRL_REG1 ACTIVE bit = 0 
              */
//              (void)FXOS8700CQ_Standby();
              /*
              ** Read the contents of the XYZ_DATA_CFG_REG
              */
//              temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
              /*
              ** Mask the bits to calculate the value to write to the XYZ_DATA_CFG_REG to enable 4g mode
              */
//              full_scale = FULL_SCALE_8G;
//              temp = (temp & (~FS_MASK)) | full_scale;
              /*
              ** Write the value back into XYZ_DATA_CFG_REG
              */
//              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              /*
              ** Set device into Active Mode
              **
              */
//              FXOS8700CQ_Active();
//              break;

            /////////////////////////////////////////////////////////////////////////////////////
//            default:
//              INPUT_ERROR = TRUE;
//              break;
//          }
//          break;

        /***************************************************************************************/
        // TO DO: Do not need it
        // (TO, Aug 2012)
//        case 'O':
         /***************************************************************************
         ** M8  : Configure oversampling mode
         ** Code Location: terminal.c
         ***************************************************************************/
          /*
          ** Enter Standby, CTRL_REG1 ACTIVE bit = 0 
          */
//          actv = FXOS8700CQ_Standby();
          /*
          ** Read the contents of the CTRL_REG2
          */
//          temp = IIC_RegRead(SlaveAddressIIC, CTRL_REG2);
//          switch (BufferRx[1])
//          {
            // Mask with Normal Mode Bits Enabled = 0x00
//            case '0': temp = (temp & ~MODS_MASK) | MOD_NORMAL;    break; 
            // Mask with Low Noise Low Power Mode Bits Enabled = 0x01
//            case '1': temp = (temp & ~MODS_MASK) | MOD_LOW_NOISE; break; 
            // Mask with High Resolution Mode Bits Enabled = 0x02
//            case '2': temp = (temp & ~MODS_MASK) | MOD_HIGH_RES;  break; 
            // Mask with Low Power Mode Bits Enabled = 0x03
//            case '3': temp = (temp & ~MODS_MASK) | MOD_LOW_POWER; break; 
//            default:  INPUT_ERROR = TRUE; break;
//          }
          //Write the value into CTRL_REG2
//          IIC_RegWrite(SlaveAddressIIC, CTRL_REG2, temp);
//          if (actv) FXOS8700CQ_Active();
//          if (INPUT_ERROR != TRUE)
//          {
            /*
            **  Echo back both ODR and HP register values
            */
//            Print_ODR_HP();
//          }
//          break;

        /***************************************************************************************/
        case 'R':
          /*
          **  Sensor register access
          */
          switch (BufferRx[1])
          {
            /////////////////////////////////////////////////////////////////////////////////////
            case 'R':
              /*
              **  RR xx  : Register Read
              */
              index = 2;
              /*
              **  Skip past space characters
              */
              while (BufferRx[index] == ' ')
              {
                index++;
              }
              /*
              **  Process hexadecimal register address input
              */
              temp = ProcessHexInput (&BufferRx[index], &value[0]);
              if (temp != 0)
              {
                index += temp;
                /*
                **  Check for end of string null
                */
                if (BufferRx[index++] == 0)
                {
                  SCISendString("= ");
                  /*
                  **  Go read register and report result
                  */
                  temp = IIC_RegRead(SlaveAddressIIC, value[0]);
                  hex2ASCII(temp,&value[0]);
                  value[2] = 0;
                  SCISendString(&value[0]);
                }
                else
                {
                  INPUT_ERROR = TRUE;
                }
              }
              else
              {
                /*
                **  If no register identified, then read all registers
                */
                SCISendString("Read All Registers\r\n");
                for (value[5]=0; value[5]<0x79;)
                {
                  hex2ASCII(value[5],&value[0]);
                  value[2] = ' ';
                  value[3] = '=';
                  value[4] = 0;
                  SCISendString(&value[0]);

                  for (value[4]=4; value[4]!=0 && value[5]<0x79; value[4]--)
                  {
                    value[0] = ' ';
                    temp = IIC_RegRead(SlaveAddressIIC, value[5]++);
                    hex2ASCII(temp,&value[1]);
                    value[3] = 0;
                    SCISendString(&value[0]);
                  }
                  SCI_putCRLF();
                }
              }
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case 'W':
              /*
              **  RW xx = nn  : Register Write
              */
              index = 2;
              /*
              **  Skip past space characters
              */
              while (BufferRx[index] == ' ')
              {
                index++;
              }
              /*
              **  Process hexadecimal register address input
              */
              temp = ProcessHexInput (&BufferRx[index], &value[0]);
              if (temp != 0)
              {
                index += temp;
                /*
                **  Check for "="
                */
                while (BufferRx[index] == ' ')
                {
                  index++;
                }
                if (BufferRx[index++] == '=')
                {
                  while (BufferRx[index] == ' ')
                  {
                    index++;
                  }
                  /*
                  **  Process hexadecimal register data input
                  */
                  temp = ProcessHexInput (&BufferRx[index], &value[1]);
                  if (temp != 0)
                  {
                    index += temp;
                    /*
                    **  Check for end of string null
                    */
                    if (BufferRx[index++] == 0)
                    {
                      /*
                      **  Go write register
                      */

                      // (TO, Aug 2012)                  
      
                      if(value[0] != CTRL_REG1) 
                      {
                        temp = FXOS8700CQ_Standby();
                      }
                      else
                      {
                        SCISendString(string_warn_ctrl_reg1);
                      }
                      IIC_RegWrite(SlaveAddressIIC, value[0], value[1]);
                      if (temp && (value[0] != CTRL_REG1))
                      {
                        FXOS8700CQ_Active();
                      }
                      /*
                      **  Go read register and verify
                      */
                      temp = IIC_RegRead(SlaveAddressIIC, value[0]);
                      if (temp == value[1])
                      {
                        SCISendString(" Success");
                        /*
                        **  Check and flag if ODR was set >= 400 Hz
                        */
                        if (value[0] == CTRL_REG1)
                        {
                          if ((temp & DR_MASK) <= DATA_RATE_400HZ)
                          {
                            // (TO, Aug 2012), no flash support for this rev
                            // USE_FLASH = TRUE;
                          }
                          else
                          {
                            USE_FLASH = FALSE;
                          }
                        }
                      }
                      else
                      {
                        SCISendString(" Fail");
                      }
                    }
                    else
                    {
                      INPUT_ERROR = TRUE;
                    }
                  }
                  else
                  {
                    INPUT_ERROR = TRUE;
                  }
                }
                else
                {
                  INPUT_ERROR = TRUE;
                }
              }
              else
              {
                INPUT_ERROR = TRUE;
              }
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            // TO DO: Do not need it
            // (TO, Aug 2012)
//            case 'O':
            /***************************************************************************
            ** RO  : Changes the output data rate
            ** Code Location: terminal.c
            ***************************************************************************/
//              index = 2;
              /*
              **  Skip past space characters
              */
//              while (BufferRx[index] == ' ')
//              {
//                index++;
//              }
              /*
              **  Accept numerical entries from '0' to '7'
              */
//              value[0] = BufferRx[index] - '0';
//              if (value[0] < 8)
//              {
                /*
                ** value[0] = 0-7
                ** 0 = 800 Hz, 1 = 400 Hz, 2 = 200 Hz, 3 = 100 Hz, 4 = 50 Hz, 5 = 12.5 Hz, 
                ** 6 = 6.25 Hz, 7 = 1.56 Hz
                */
//                value[0] <<= 3;
                /*
                ** Place device into Standby mode
                */
//                temp = FXOS8700CQ_Standby();
                /*
                ** Write the value[0] into CTRL_REG1, masked with the selected output data rate
                */
//                IIC_RegWrite(SlaveAddressIIC, CTRL_REG1,
//                             (IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & (~DR_MASK)) | value[0]);
//                if (temp)
//                {
//                   FXOS8700CQ_Active();
//                }
                /*
                **  Echo back both ODR and HP register values
                */
//                Print_ODR_HP();
//              }
//              else
//              {
//                INPUT_ERROR = TRUE;
//              }
//              break;

            /////////////////////////////////////////////////////////////////////////////////////
            // TO DO: Do not need it
            // (TO, Aug 2012)
//            case 'H':
            /***************************************************************************
            ** RH  : Changes the high pass filter cutoff frequency
            ** Code Location: terminal.c
            ***************************************************************************/
//              index = 2;
              /*
              **  Skip past space characters
              */
//              while (BufferRx[index] == ' ')
//              {
//                index++;
//              }
              /*
              **  Accept numerical entries from '0' to '3'
              */
//              value[0] = BufferRx[index] - '0';
              //Put device into standby mode
//              temp = FXOS8700CQ_Standby();
              // Write the selected cutoff frequency masked with the contents of the HP_FILTER_CUTOFF_REG
              // Options are 0x00-0x03
//              IIC_RegWrite(SlaveAddressIIC, HP_FILTER_CUTOFF_REG, 
//                (IIC_RegRead(SlaveAddressIIC, HP_FILTER_CUTOFF_REG) & (~SEL_MASK)) | value[0]);
//              if (temp)
//              {
                // Put device into active mode
//                FXOS8700CQ_Active();
//              }
              /*
              **  Echo back both ODR and HP register values
              */
//              Print_ODR_HP();
             
//              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case 'F':
              /*
              **  RF  : Report ODR and HP Frequencies
              */
              Print_ODR_HP();
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            default:            
              INPUT_ERROR = TRUE;
              break;
          }
          break;

        /***************************************************************************************/
        case 'C': // Read and display current XYZ data as signed counts
        case 'G': // Read and display current XYZ data as signed g's
          index = 1;
          /*
          **  Skip past space characters
          */
          while (BufferRx[index] == ' ')
          {
            index++;
          }
          /*
          **  Select High pass or normal output data
          */
          temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
          value[0] = temp;
          switch (BufferRx[index])
          {
            case 'N': temp &= ~HPF_OUT_MASK; break; // N = Normal XYZ data
            case 'H':  // H = HPF XYZ data
                temp |=  HPF_OUT_MASK;
              break;
            case  0: break;                         //   = repeat same type as last reading
            default:  INPUT_ERROR = TRUE; break;
          }
          
          if (INPUT_ERROR != TRUE)
          {
            // Check is the device is in standby mode
            actv = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK;
          
            /*
            **  Need to put the device into Standby to change the HPF mode
            */
            if (((temp ^ value[0]) & HPF_OUT_MASK))
            {
              /*
              **  Change HPF filter mode
              */
              (void)FXOS8700CQ_Standby();
              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              actv = 0;
            }
            
            // (TO, Aug 2012)
            CheckHPF(0); // Warn if restarting the HPF filter
            
            /*
            **  Activate the device in case it was in Standby or the HPF was updated
            */
            if (!actv)
              FXOS8700CQ_Active();
            
            /*
            **  Wait for next sample to be acquired
            */
            value[0] = (IIC_RegRead( SlaveAddressIIC, M_CTRL_REG1 )) & 
              (M_HMS1_MASK | M_HMS0_MASK);  

            if( (value[0] == HYBRID_ACTIVE) || (value[0] == ACCEL_ACTIVE) )
              while (!(IIC_RegRead(SlaveAddressIIC, STATUS_00_REG) & ZYXDR_MASK));
            else
              while (!(IIC_RegRead(SlaveAddressIIC, M_DR_STATUS_REG) & ZYXDR_MASK));
              
            /*
            **  Print out the data in the requested format
            */
            IIC_RegReadN(SlaveAddressIIC, OUT_X_MSB_REG, 6, &value[0]);
            IIC_RegReadN(SlaveAddressIIC, M_OUT_X_MSB_REG, 6, &value[6]);
            CopyXYZ(&value[0]);
            SCI_putCRLF(); // TO APR2013
            if (BufferRx[0] == 'G')
              PrintXYZfrac();
            else
              PrintXYZdec();
          }
          break;
        /***************************************************************************************/
//         case 'P': // Read and display current XYZ data as signed g's and orientation (P/L)
//          index = 1;
          /*
          **  Skip past space characters
          */
//          while (BufferRx[index] == ' ')
//          {
//            index++;
//          }
          /*
          **  Select High pass or normal output data
          */
//          temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
//          value[0] = temp;
//          switch (BufferRx[index])
//          {
//            case 'N': temp &= ~HPF_OUT_MASK; break; // N = Normal XYZ data
//            case 'H':  // H = HPF XYZ data
//                temp |=  HPF_OUT_MASK;
//              break;
//            case  0: break;                         //   = repeat same type as last reading
//            default:  INPUT_ERROR = TRUE; break;
//          }
          
//          if (INPUT_ERROR != TRUE)
//          {
            // Check is the device is in standby mode
//            actv = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK;
          
            /*
            **  Need to put the device into Standby to change the HPF mode
            */
//            if (((temp ^ value[0]) & HPF_OUT_MASK))
//            {
              /*
              **  Change HPF filter mode
              */
//              (void)FXOS8700CQ_Standby();
//              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
//              actv = 0;
//            }
            
//            CheckHPF(1); // Warn if restarting the HPF filter
            
            /*
            **  Activate the device in case it was in Standby or the HPF was updated
            */
//            if (!actv)
//              FXOS8700CQ_Active();
            
            /*
            **  Wait for next sample to be acquired
            */
//            while (!(IIC_RegRead(SlaveAddressIIC, STATUS_00_REG) & ZYXDR_MASK))
              ;
            /*
            **  Print out the data in the requested format
            */
//            IIC_RegReadN(SlaveAddressIIC, OUT_X_MSB_REG, 6, &value[0]);
//            CopyXYZ(&value[0]);
//            if (BufferRx[0] == 'P')
//              PrintXYZfrac();
//            else
//              PrintXYZdec();
            
             /*
                  **  Read Portrait/Landscape
                  */
//                  IIC_RegWrite(SlaveAddressIIC, PL_CFG_REG, 0x40);
//                  temp = IIC_RegRead(SlaveAddressIIC, PL_STATUS_REG);
//                  hex2ASCII(temp,&value[0]);
//                  value[2] = 0;
//                  SCISendString(" P= ");
//                  SCISendString(&value[0]);
//          }
//          break;
        /***************************************************************************************/
        case 'F': //  Stream XYZ data via FIFO interrupts
        case 'S': //  Stream XYZ data
        case 'I': //  Stream XYZ data via interrupts
          StreamMode.Byte = 0;
          /*
          **  Skip past space characters
          */
          index = 1;
          while (BufferRx[index] == ' ')
            index++;

          /*
          **  Select output data format: counts or g's
          */
          switch (BufferRx[index++])
          {
            case 'C': StreamMode.Byte = STREAM_C_MASK; break; // C : Stream as signed counts
            case 'G': StreamMode.Byte = STREAM_G_MASK; break; // G : Stream as signed g's
            default: INPUT_ERROR = TRUE; break;
          }

          /*
          **  Select High pass or normal output data
          */
          if (!INPUT_ERROR)
          {
            temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
            value[0] = temp;
            switch (BufferRx[index++])
            {
              case 'N': temp &= ~HPF_OUT_MASK; break; // N = Normal XYZ data
              case 'H':  // H = HPF XYZ data
                  temp |=  HPF_OUT_MASK;
                break;
              case  0: break;                         //   = repeat same type as last reading
              default:  INPUT_ERROR = TRUE; break;
            }
          }
          /*
          **  Skip past space characters
          */
          while (BufferRx[index] == ' ')
            index++;
            
          /*
          **  Select Interrupt number for Interrupt based streaming
          */
          if (!INPUT_ERROR && BufferRx[0]=='I')
          {
            /*
            **  Set INT number
            */
            switch (BufferRx[index]) {
              //case '1' : value[1] = 0xFD; break; // Direct interrupts to INT1
              // (TO, Aug 2012)
              case '1' : value[1] = 0x01; break; // Direct interrupts to INT1
              case '2' : value[1] = 0x00; break; // Direct interrupts to INT2
              default  : INPUT_ERROR = TRUE; break; // Invalid INT number
            }
          }
                    
          /*
          **  Get the desired FIFO watermark. (defaulting to the current watermark setting)
          */
          if (!INPUT_ERROR && BufferRx[0]=='F')
          {
            value[1] = IIC_RegRead(SlaveAddressIIC, F_SETUP_REG) & F_WMRK_MASK;
            // (TO, Aug 2012) why is this here? it is interfering with fifo hpf...
            // TO DO: verify
            //temp = value[1];
                        
            if (isnum (BufferRx[index]) == TRUE)
            {
              value[1] = BufferRx[index++] - '0';
              if (isnum (BufferRx[index]) == TRUE)
              {
                value[1] *= 10;
                value[1] += (BufferRx[index++] - '0');
              }
            }
            // Check that the watermark value is valid
            if (value[1] > 31)
              INPUT_ERROR = TRUE;
            
            // (TO, Aug 2012)
            // Check if the part is configured for hybrid or accel only mode.
            // FIFO streaming will not work in mag only mode since FIFO 
            // is only for accel.

            value[4] = IIC_RegRead(SlaveAddressIIC, M_CTRL_REG1) & M_HMS_MASK;

            SCISendString( "\r\n* !!! - FIFO Streaming is not possible in Mag Only mode *\r\n" );

            if( value[4] == MAG_ACTIVE ) 
            {
              INPUT_ERROR = TRUE;
            }
          }
          
          /*
          **  Activate streaming mode if there were no input errors
          */
          if (!INPUT_ERROR)
          {
            // Check is the device is in active or standby mode
            actv = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK;
            
            /*
            **  Need to reactivate the device if:
            **    - the HPF mode changed
            **    - Interrupt of FIFO mode (because both need to set CTRL registers)
            **    - the FIFO watermark value changed
            */
            if (((temp ^ value[0]) & HPF_OUT_MASK) || (BufferRx[0]!='S'))
            {
              /*
              **  Enter Standby mode
              */
              (void)FXOS8700CQ_Standby();
              /*
              **  Change HPF filter mode
              */
              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              actv = 0;
            }
            
            /*
            **  Display streaming configuration
            */

            // (TO, Aug 2012)
            
            SCISendString(string_warn_slow_streaming);
            SCISendString(string_Streaming);
 
            if (BufferRx[0]=='I') // using Interrupts
            {  
              SCISendString(string_Interrupts);
              SCI_CharOut(BufferRx[index]);
            } else if (BufferRx[0]=='F') // using the FIFO
              SCISendString(string_FIFO);
            if (STREAM_C == TRUE)
              SCISendString(string_Counts);
            else
              SCISendString(string_Gs);
            
            CheckHPF(0);  // Warn if restarting the HPF filter
            //Check800Hz(); // Warn that 800 Hz data may not be stored correctly in flash
            
            /*
            **  Setup streaming operation
            */
            PROMPT_WAIT = TRUE;
            if (BufferRx[0]=='S') // Streaming XYZ data
            {
              functional_block = FBID_XYZ_SAMPLE;
              XYZ_STREAM  = TRUE;
              POLL_ACTIVE = TRUE;
              
              // TO APR2013
              FXOS8700CQ_Standby();
              actv = FALSE;
              
              // disable all interrupt outputting for streaming
              
              IIC_RegWrite(SlaveAddressIIC, CTRL_REG4, 0x00);
              
            }
            else // FIFO and Streaming modes require interrupts
            {
              
              // TO APR2013
              FXOS8700CQ_Standby();
              actv = FALSE;
              
              //IIC_RegWrite(SlaveAddressIIC, CTRL_REG3, PP_OD_MASK);
              // (TO, Aug 2012)
              //IIC_RegWrite(SlaveAddressIIC, CTRL_REG3, 0);
              
              if (BufferRx[0]=='I') // Stream data via Interrupts
              {  
                functional_block = FBID_XYZ_SAMPLE;
                /*
                **  Activate sensor interrupts
                **  - enable Data Ready interrupt
                **  - route Data Ready to selected INT
                */
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG4, INT_EN_DRDY_MASK);
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG5, value[1]);
              }
              else // Stream data via FIFO
              {  
                functional_block = FBID_FIFO;
                /*
                **  Set the FIFO mode to circular
                **  this does not need to be in standby mode unless the watermark changes
                */
                IIC_RegWrite(SlaveAddressIIC, F_SETUP_REG, F_MODE_CIRCULAR | value[1]);
                            
                /*
                **  Activate sensor interrupts
                **  - enable FIFO interrupt
                **  - route FIFO interrupt to INT2
                */
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG4, INT_EN_FIFO_MASK);
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG5, 0);
                               
              }
              /*
              **  Configure the interrupt pins for falling edge and open drain output
              */
              INT_STREAM = TRUE;
              /*
              switch( value[1] ) 
              {
                case '1':
                  INT1_FALLING;
                  break;
                case '2':
                  INT2_FALLING;
                  break;
                default:
                  break;
              }
              */

              INT_BOTH_FALLING;              
              // Clear and Enable MC9S08 Interrupts
              InterruptsActive ();
            }
            /*
            **  Activate the device in case it was in Standby or the HPF/Watermark changed
            */
            
            if (!actv)
            FXOS8700CQ_Active();
            // (TO, Aug 2012), read and discard first sample to make sure INT pin
            // goes high  
            IIC_RegReadN(SlaveAddressIIC, OUT_X_MSB_REG, 6, &value[0]);
            IIC_RegReadN(SlaveAddressIIC, M_OUT_X_MSB_REG, 6, &value[6]);
            
            /*
            **  Reset FIFO group ID
            */
            value[5] = 0;
            
          }
          break;


        /***************************************************************************************/
        default:
          /*
          **  Undefined inputs are ignored.
          */
          INPUT_ERROR = TRUE;
          break;
      }
      if (INPUT_ERROR == TRUE)
      {
        SCISendString(string_What);
      }
      SCI_INPUT_READY = FALSE;
      temp = 0;
    }
    /*
    **  Data streaming is stopped by any character input.
    */
    else
    {
      /*
      **  Turn off data streaming, TO DO: is this working??? (TO, Aug 2012)
      */
      INT_PINS_DISABLED;
      POLL_ACTIVE = FALSE;
      XYZ_STREAM = FALSE;
      INT_STREAM = FALSE;
      SCI_INPUT_READY = FALSE;
      PROMPT_WAIT = FALSE;

      // Disable the FIFO in case it was using the FIFO
      value[2] = IIC_RegRead(SlaveAddressIIC, F_SETUP_REG) & F_WMRK_MASK;
      IIC_RegWrite(SlaveAddressIIC, F_SETUP_REG, value[2]);
      
      /*
      **  If samples were being buffered in FLASH, output them now.
      */
      
      // (TO, Aug 2012) no flash...

//      if (USE_FLASH == TRUE)
//      {
//        SCI_putCRLF();
//        address_out[0] = 0;
//        address_out[1] = 0;
//        address_out[2] = 0;
        /*
        **  Read samples stored in Data Flash
        */
//        while (USE_FLASH == TRUE)
//        {
          // (TO, Aug 2012)
          //ReadDataFlashXYZ();
          
          /*
          **  Output formats are:
          **    - Stream XYZ data as signed counts
          **    - Stream XYZ data as signed g's
          */
//          if (STREAM_C == 1)
//            PrintXYZdec();
//          else
//            PrintXYZfrac();
//          SCI_putCRLF();
          /*
          **  Adjust Data Flash address pointer
          */
//          if (address_out[2] > 245)
//          {
//            address_out[2] = 0;
//            address_out[1]++;
//          }
//          else
//          {
//            address_out[2] += 6;
//          }
          /*
          **  Check if we're done.
          */
//          if (address_out[0] >= address_in[0])
//          {
//            if (address_out[1] >= address_in[1])
//            {
//              if (address_out[2] >= address_in[2])
//              {
//                USE_FLASH = FALSE;
//              }
//            }
//          }
//        }
        /*
        **  Erase the Data Flash
        */
//        USE_FLASH = TRUE;
//        SCISendString  ("\r\nErasing Data Flash ... Please wait ");
//        while (SCItxActive);
//        ClearDataFlash();
//      }
      functional_block = FBID_MAX;
    }
  }
}


/*********************************************************\
**  Terminal Output
\*********************************************************/
void OutputTerminal (byte BlockID, byte *ptr)
{
  
  /*
  SCISendString( "X acc reg= " );  
  SCI_ByteOut( ptr[0] );
  SCISendString( ", " );
  SCI_ByteOut( ptr[1] );
  SCISendString( ", " );
  SCISendString( "Y acc reg= " );  
  SCI_ByteOut( ptr[2] );
  SCISendString( ", " );
  SCI_ByteOut( ptr[3] );
  SCISendString( ", " );
  SCISendString( "Z acc reg= " );  
  SCI_ByteOut( ptr[4] );
  SCISendString( ", " );
  SCI_ByteOut( ptr[5] );
  SCISendString( ", " );
  SCISendString( "X mag reg= " );  
  SCI_ByteOut( ptr[6] );
  SCISendString( ", " );
  SCI_ByteOut( ptr[7] );
  SCISendString( ", " );
  SCISendString( "Y mag reg= " );  
  SCI_ByteOut( ptr[8] );
  SCISendString( ", " );
  SCI_ByteOut( ptr[9] );
  SCISendString( ", " );
  SCISendString( "Z mag reg= " );  
  SCI_ByteOut( ptr[10] );
  SCISendString( ", " );
  SCI_ByteOut( ptr[11] ); 
  SCI_putCRLF(); 
  */

  switch (BlockID)
  {
    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_XYZ_SAMPLE:
      /*
      **  XYZ Sample Registers (0x01 - 0x06)
      */
      CopyXYZ(ptr);
      /*
      **  If the ODR >= 400Hz, buffer the samples in the Data Flash.
      */
      if (USE_FLASH == TRUE)
      {
        /*
        **  Provide a visual indication that we're buffering and write to Flash
        */
        // (TO, Aug 2012)
        //WriteFlashBuffer1(); // First half of write to flash
        PrintBuffering();
        //WriteFlashBuffer2(); // Second half of write to flash
      }
      else
      {
        /*
        **  Output formats are:
        **    - Stream XYZ data as signed counts
        **    - Stream XYZ data as signed g's
        */
        if (STREAM_C == 1)
          PrintXYZdec();
        else
          PrintXYZfrac();
        SCI_putCRLF();
      }
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_LANDSCAPE_PORTRAIT:
      /*
      **  Landscape/Portrait Orientation Status Register (0x18)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_FREEFALL_MOTION_1:
      /*
      **  Freefall/Motion 1 Source Register (0x24)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_TRANSIENT:
      /*
      **  Transient Source Register (0x2C)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_PULSE:
      /*
      **  Pulse Source Register (0x30)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_SYSTEM_MODE:
      /*
      **  System Mode Register (0x14)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_FIFO:
      /*
      **  FIFO
      **
      **  If the ODR >= 400Hz, buffer the samples in the Data Flash.
      */
      if (USE_FLASH == TRUE)
      {
        /*
        **  Provide a visual indication that we're buffering
        */
        PrintBuffering();        
        /*
        **  Save the samples in the FIFO buffer into the Data Flash
        */
        value[0] = RegisterFlag.Byte & F_CNT_MASK;
        for (value[1]=0; value[1]!=value[0]; value[1]++)
        {
          /*
          **  Save sample to Data Flash
          */
          CopyXYZ(&fifo_data[value[1]].Sample.XYZ.x_msb);
          // (TO, Aug 2012)
          //WriteFlashBuffer1(); // First half of write to flash
          //WriteFlashBuffer2(); // Second half of write to flash
        }
      }
      else
      {
        PrintFIFO();
      }
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    default:
      break;
  }
}


/***********************************************************************************************\
* Private functions
\***********************************************************************************************/

/*********************************************************\
* Print a warning if trying to store data in Flash at 800 Hz
\*********************************************************/

// TO, Aug 2012

//void Check800Hz (void)
//{
  /*
  **  Streaming mode at 800 Hz is unreliable because the flash is too slow
  */
//  if ((IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & DR_MASK) == DATA_RATE_800HZ) {
//    SCI_putCRLF();
//    SCISendString(string_warn_banner);
//    SCISendString(string_warn_slow_flash);
//    SCISendString(string_warn_banner);
//    SCI_putCRLF();
//  }
//}

/*********************************************************\
* Print a warning if the HPF is initializing
\*********************************************************/
void CheckHPF (byte spacer)
{
  /*
  **  HPF starts near normal XYZ data values after an activate() and then
  **  takes a number of samples to settle to the correct value
  */
  if (((IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK) == 0) &&
      ((IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG) & HPF_OUT_MASK)) )       
  {
    SCI_putCRLF();
    SCISendString(string_warn_banner);
    SCISendString(string_warn_hpf);
    SCISendString(string_warn_banner);
    SCI_putCRLF();
    if (spacer)
      SCISendString(string_prompt_spacer);  
  }
}

/*********************************************************\
* Print accelerometer's OverSample, ODR, HP and Mode
\*********************************************************/
void Print_ODR_HP (void)
{
  
  /* Print operating mode: mag only, accel only or hybrid (TO, Aug 2012) */
  
  SCI_putCRLF();
  SCISendString("Operating Mode = ");
  value[0] = IIC_RegRead(SlaveAddressIIC, M_CTRL_REG1) & HYBRID_ACTIVE;
  SCISendString(OperatingModevalues[value[0]]);
  SCISendString(", ");

  // (TO, Aug 2012), indicate ACTIVE/STBY mode

  if( (IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK) == TRUE )
  {
    SCISendString("ACTIVE");
  }
  else 
  {
    SCISendString("STANDBY");
  }

  SCISendString(", ");
  SCI_putCRLF();
  
  /* Report ODR */

  SCISendString("ODR = ");
  value[1] = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & DR_MASK;
  
  // ODR is different based on operating mode (TO, Aug 2012)
  
  if( value[0] == HYBRID_ACTIVE ) 
  {
    SCISendString(HybridODRvalues[value[1] >> 3]);
    
  } 
  else 
  {
    SCISendString(ODRvalues[value[1] >> 3]); 
  }     
  
  SCISendString("Hz, ");
  SCI_putCRLF(); 
  
  /*
  **  Use onboard flash to store XYZ data if ODR >= 400 Hz
  */  
  
  // Take into account hybrid mode, ODRs divide by 2 (TO, Aug 2012)
  // do not support flash at the moment.
  /*
  if ( ((value[1] <= DATA_RATE_400HZ) && (value[0] != HYBRID_ACTIVE)) ||
    ((value[1] == DATA_RATE_800HZ) && (value[0] == HYBRID_ACTIVE)) )
  {
    USE_FLASH = TRUE;
  }
  else
  {
    USE_FLASH = FALSE;
  }       
  */

  /* if in hyrid mode see if hybrid auto increment feature is enabled.
   * (TO, Aug 2012) */

  if( value[0] == HYBRID_ACTIVE ) 
  {
    SCISendString("Hybrid auto inc = ");
    value[2] = IIC_RegRead(SlaveAddressIIC, M_CTRL_REG2) & M_HYB_AUTOINC_MASK;
    if(value[2]) 
    {
      SCISendString("ON, ");
    } 
    else 
    {
      SCISendString("OFF, ");  
    }
    SCI_putCRLF();
  }
    
  /* Report Mag OSR (TO, Aug 2012) */ 

  SCISendString("Mag OSR = ");
  value[2] = IIC_RegRead(SlaveAddressIIC, M_CTRL_REG1) & M_OSR_MASK;
  // right justify mag OSR fields
  value[2] = value[2] >> 2;
  // right justify ODR fields 
  value[1] >>= 3;
  // half for hybrid mode (TO, Aug 2012)
  value[4] = MagOSRLookUp[value[1]][value[2]];
  SCISendString(MagOSRvalues[ value[4] ] );
  SCISendString(", ");
  SCI_putCRLF(); 
  
  SCISendString("Accel OSR = ");
  value[4] = IIC_RegRead(SlaveAddressIIC, CTRL_REG2) & MODS_MASK;
  SCISendString(AccelOSRModevalues[value[4]]);
  SCISendString(", ");
  SCI_putCRLF();   

  SCISendString("HP = ");
  value[2] = HPFlookup[value[4]][value[1]];
  value[2] += IIC_RegRead(SlaveAddressIIC, HP_FILTER_CUTOFF_REG) & SEL_MASK;
  // adjust for hybrid mode
  if( value[0] == HYBRID_ACTIVE )
  {
    value[2] = value[2] + 1;
  }
  SCISendString(HPvalues[value[2]]);
  SCISendString("Hz, ");
  SCI_putCRLF();

  SCISendString("Accel FSR = ");
  
  // (TO, Aug 2012), nevermind about Stby now that it is indicated above
  //value[3] = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK;
  
  //if (value[3])
  //{
    value[3] = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
    value[3] &= FS_MASK;
  //  value[3]++;
  //}
  SCISendString(GRange[value[3]]);
}


/*********************************************************\
*
\*********************************************************/

//void WriteFlashBuffer1 (void)
//{
//  /*
//  **  Enable SO as a busy indicator
//  */
//  SPI_SS_SELECT;
//  SPI_ChrShift(0x70);
//  SPI_SS_DESELECT;
//  /*
//  **  Start multibyte write to Data Flash
//  */
//  // (TO, Aug 2012)
//  // DATAFLASH_WriteEnableLatch();
//  SPI_SS_SELECT;
//  SPI_ChrShift(0xAD);
//  SPI_ChrShift(address_in[0]);
//  SPI_ChrShift(address_in[1]);
//  SPI_ChrShift(address_in[2]);
//  SPI_ChrShift(x_acc_value.Byte.hi);
//  SPI_ChrShift(x_acc_value.Byte.lo);
//  SPI_SS_DESELECT;
  /*
  **  Wait for completion of the first write
  */
//  SPI_SS_SELECT;
//  while (SPI_SO==0);
//  SPI_SS_DESELECT;
  /*
  **  Send next 2 bytes
  */
//  SPI_SS_SELECT;
//  SPI_ChrShift(0xAD);
//  SPI_ChrShift(y_acc_value.Byte.hi);
//  SPI_ChrShift(y_acc_value.Byte.lo);
//  SPI_SS_DESELECT;
//}


/*********************************************************\
*
\*********************************************************/
//void WriteFlashBuffer2 (void)
//{
  /*
  **  Wait for completion of the first write
  */
//  SPI_SS_SELECT;
//  while (SPI_SO==0); // Wait for the Flash to complete
//  SPI_SS_DESELECT;
  /*
  **  Send last 2 bytes
  */
//  SPI_SS_SELECT;
//  SPI_ChrShift(0xAD);
//  SPI_ChrShift(z_acc_value.Byte.hi);
//  SPI_ChrShift(z_acc_value.Byte.lo);
//  SPI_SS_DESELECT;
  /*
  **  Adjust Data Flash address pointer
  */
//  if (address_in[2] > 245)
//  {
//    address_in[2] = 0;
//    address_in[1]++;
//  }
//  else
//  {
//    address_in[2] += 6;
//  }
  /*
  **  Wait for completion of the first write
  */
//  SPI_SS_SELECT;
//  while (SPI_SO==0); // Wait for the Flash to complete
//  SPI_SS_DESELECT;
  /*
  **  Send WRDI to end the previous Write
  */
//  SPI_SS_SELECT;
//  SPI_ChrShift(0x04);
//  SPI_SS_DESELECT;
  /*
  **  Disable SO as a busy indicator
  */
//  SPI_SS_SELECT;
//  SPI_ChrShift(0x80);
//  SPI_SS_DESELECT;
//}


/*********************************************************\
*
\*********************************************************/
void PrintBuffering (void)
{
  if (temp > 50)
  {
    SCI_putCRLF();
    temp = 0;
    address_out[2] += 50;
    if (address_out[2] < 50)
    {
      address_out[1]++;
     if (address_out[1] == 0)
      {
        address_out[0]++;
      }
    }
  }
  if (temp++ == 0)
  {
    hex2ASCII(address_out[0],&value[0]);
    hex2ASCII(address_out[1],&value[2]);
    value[4] = 0;
    SCISendString(&value[0]);
    hex2ASCII(address_out[2],&value[0]);
    value[2] = 0;
    SCISendString(&value[0]);
  }
  SCI_CharOut('*');
}


/*********************************************************\
*
\*********************************************************/
void PrintFIFO (void)
{
  /*
  **  Identify if this is either a FIFO Overflow or Watermark event
  */
  if (RegisterFlag.F_OVF_BIT == 1)
  {
    SCISendString  ("FIFO Overflow ");
  }
  if (RegisterFlag.F_WMRK_FLAG_BIT == 1)
  {
    SCISendString  ("FIFO Watermark ");
  }
  /*
  **  Display FIFO count
  */
  value[0] = RegisterFlag.Byte & F_CNT_MASK;
  SCISendString("Samples= ");
  SCI_Dec_Out(value[0],0);
  /*
  **  Identify FIFO group
  */
  SCISendString (" group= ");
  hex2ASCII(value[5],&value[1]);
  value[3] = 0;
  value[5]++;
  SCISendString(&value[1]);
  SCI_putCRLF();
  /*
  **  Output results
  */
  for (value[1]=0; value[1]!=value[0]; value[1]++)
  {
    CopyXYZ(&fifo_data[value[1]].Sample.XYZ.x_msb);
    /*
    **  Output formats are:
    **    - Stream XYZ data as signed counts
    **    - Stream XYZ data as signed g's
    */
    if (STREAM_C == 1)
    {
      PrintXYZdec();
    }
    else
    {
      PrintXYZfrac();
    }
    SCI_putCRLF();
  }
}


/*********************************************************\
*
\*********************************************************/
/*
void ReadDataFlashXYZ (void)
{
  SPI_SS_SELECT;
  SPI_ChrShift(0x0B);
  SPI_ChrShift(address_out[0]);
  SPI_ChrShift(address_out[1]);
  SPI_ChrShift(address_out[2]);
  SPI_ChrShift(0x00);
  x_acc_value.Byte.hi = SPI_ChrShiftR(0x00);
  x_acc_value.Byte.lo = SPI_ChrShiftR(0x00);
  y_acc_value.Byte.hi = SPI_ChrShiftR(0x00);
  y_acc_value.Byte.lo = SPI_ChrShiftR(0x00);
  z_acc_value.Byte.hi = SPI_ChrShiftR(0x00);
  z_acc_value.Byte.lo = SPI_ChrShiftR(0x00);

  x_mag_value.Byte.hi = SPI_ChrShiftR(0x00);
  x_mag_value.Byte.lo = SPI_ChrShiftR(0x00);
  y_mag_value.Byte.hi = SPI_ChrShiftR(0x00);
  y_mag_value.Byte.lo = SPI_ChrShiftR(0x00);
  z_mag_value.Byte.hi = SPI_ChrShiftR(0x00);
  z_mag_value.Byte.lo = SPI_ChrShiftR(0x00);
  SPI_SS_DESELECT;
}
*/

/*********************************************************\
*
\*********************************************************/
byte ProcessHexInput (byte *in, byte *out)
{
  byte data;

  data = *in++;
  if (ishex(data) == TRUE)
  {
    data = tohex(data);
  }
  else
  {
    return (0);
  }
  if (ishex(*in) == TRUE)
  {
    data <<= 4;
    data += tohex(*in);
    *out = data;
    return (2);
  }
  else if ((*in == ' ') || (*in == 0))
  {
    *out = data;
    return (1);
  }
  return (0);
}


/*********************************************************\
*
\*********************************************************/

// (TO, Aug 2012)
void CopyXYZ (byte *ptr)
{
  x_acc_value.Byte.hi = *ptr++;
  x_acc_value.Byte.lo = *ptr++;
  y_acc_value.Byte.hi = *ptr++;
  y_acc_value.Byte.lo = *ptr++;
  z_acc_value.Byte.hi = *ptr++;
  z_acc_value.Byte.lo = *ptr++;

  // right align accel values since 2 LSBs are garbage
  //x_acc_value.Word = (Word)((sWord)x_acc_value.Word >> 2);
  //y_acc_value.Word = (Word)((sWord)y_acc_value.Word >> 2);
  //z_acc_value.Word = (Word)((sWord)z_acc_value.Word >> 2);

  x_mag_value.Byte.hi = *ptr++;
  x_mag_value.Byte.lo = *ptr++;
  y_mag_value.Byte.hi = *ptr++;
  y_mag_value.Byte.lo = *ptr++;
  z_mag_value.Byte.hi = *ptr++;
  z_mag_value.Byte.lo = *ptr++;
         
}


/*********************************************************\
*
\*********************************************************/
// (TO, Aug 2012)
void PrintXYZdec8 (void)
{
  SCISendString("X acc = ");
  SCI_s8dec_Out(x_acc_value);
  SCISendString("; Y acc = ");
  SCI_s8dec_Out(y_acc_value);
  SCISendString("; Z acc= ");
  SCI_s8dec_Out(z_acc_value);

  SCISendString("; X mag = ");
  SCI_s8dec_Out(x_mag_value);
  SCISendString("; Y mag = ");
  SCI_s8dec_Out(y_mag_value);
  SCISendString("; Z mag= ");
  SCI_s8dec_Out(z_mag_value);
}


/*********************************************************\
*
\*********************************************************/
// (TO, Aug 2012)
/*
void PrintXYZdec10 (void)
{
  SCISendString("X acc= ");
  SCI_s10dec_Out(x_acc_value);
  SCISendString("; Y acc=");
  SCI_s10dec_Out(y_acc_value);
  SCISendString("; Z acc=");
  SCI_s10dec_Out(z_acc_value);

  SCISendString("; X mag= ");
  SCI_s10dec_Out(x_mag_value);
  SCISendString("; Y mag=");
  SCI_s10dec_Out(y_mag_value);
  SCISendString("; Z mag=");
  SCI_s10dec_Out(z_mag_value);
}
*/

/*********************************************************\
*
\*********************************************************/
// (TO, Aug 2012)
/*
void PrintXYZdec12 (void)
{
  SCISendString("X acc= ");
  SCI_s12dec_Out(x_acc_value);
  SCISendString("; Y acc=");
  SCI_s12dec_Out(y_acc_value);
  SCISendString("; Z acc=");
  SCI_s12dec_Out(z_acc_value);

  SCISendString("; X mag= ");
  SCI_s12dec_Out(x_mag_value);
  SCISendString("; Y mag=");
  SCI_s12dec_Out(y_mag_value);
  SCISendString("; Z mag=");
  SCI_s12dec_Out(z_mag_value);
}

*/


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec14 (void)
{ 
  
  SCISendString("X acc= ");
  SCI_sDec_Out(x_acc_value, 16);
  //SCI_s14dec_Out(x_acc_value);
  SCISendString("; Y acc=");
  SCI_sDec_Out(y_acc_value, 16);
  //SCI_s14dec_Out(y_acc_value);
  SCISendString("; Z acc=");
  SCI_sDec_Out(z_acc_value, 16);
  //SCI_s14dec_Out(z_acc_value);

  SCISendString("; X mag= ");
  SCI_sDec_Out(x_mag_value, 16);
  SCISendString("; Y mag=");
  SCI_sDec_Out(y_mag_value, 16);
  SCISendString("; Z mag=");
  SCI_sDec_Out(z_mag_value, 16);
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec (void)
{
  
  // (TO, Aug 2008)

  // determine device mode (TO, Aug 2012)

  byte mode = (IIC_RegRead( SlaveAddressIIC, M_CTRL_REG1 )) & 
    (M_HMS1_MASK | M_HMS0_MASK);  

  if( (mode == HYBRID_ACTIVE) || (mode == ACCEL_ACTIVE) )
  {  

    SCISendString("Xacc=");
    SCI_s14dec_Out(x_acc_value);
    SCISendString(";Yacc=");
    SCI_s14dec_Out(y_acc_value);
    SCISendString(";Zacc=");
    SCI_s14dec_Out(z_acc_value);
  }

  // (TO, Aug 2008)
  // keep FIFO Streaming exclusive to accelerometer
  if( ((mode == HYBRID_ACTIVE) || (mode == MAG_ACTIVE)) &&
    (functional_block != FBID_FIFO)  )
  {

    SCISendString(";Xmag=");
    SCI_sDec_Out(x_mag_value, 0);
    SCISendString(";Ymag=");
    SCI_sDec_Out(y_mag_value, 0);
    SCISendString(";Zmag=");
    SCI_sDec_Out(z_mag_value, 0);
  }

}


/*********************************************************\
*
\*********************************************************/
void PrintXYZfrac (void)
{
  
  // determine device mode (TO, Aug 2012)

  byte mode = (IIC_RegRead( SlaveAddressIIC, M_CTRL_REG1 )) & 
    (M_HMS1_MASK | M_HMS0_MASK);  

  if( (mode == HYBRID_ACTIVE) || (mode == ACCEL_ACTIVE) )
  {                                    
      
    SCISendString("Xacc=");
    SCI_s14frac_Out(x_acc_value);
    SCISendString(";Yacc=");
    SCI_s14frac_Out(y_acc_value);
    SCISendString(";Zacc=");
    SCI_s14frac_Out(z_acc_value);
  
  }

  // (TO, Aug 2008)
  // keep FIFO Streaming exclusive to accelerometer
  if( ((mode == HYBRID_ACTIVE) || (mode == MAG_ACTIVE)) &&
    (functional_block != FBID_FIFO)  )
  {

    x_mag_value.Word = (word)((signed short)x_mag_value.Word / 10);
    SCISendString(";Xmag=");
    SCI_sDec_Out(x_mag_value, 0);
    SCISendString("uT");
    y_mag_value.Word = (word)((signed short)y_mag_value.Word / 10);
    SCISendString(";Ymag=");
    SCI_sDec_Out(y_mag_value, 0);
    SCISendString("uT");
    z_mag_value.Word = (word)((signed short)z_mag_value.Word / 10);
    SCISendString(";Zmag=");
    SCI_sDec_Out(z_mag_value, 0);
    SCISendString("uT");

  }
}


/*********************************************************\
*
\*********************************************************/
void ClearDataFlash (void)
{
  address_in[0] = 0;
  address_in[1] = 0;
  address_in[2] = 0;
  address_out[0] = 0;
  address_out[1] = 0;
  address_out[2] = 0;
  // (TO, Aug 2012)
  //DATAFLASH_Erase();
}

